const template = document.createElement('template');
template.innerHTML = `
<div>
  Hello, <span id="name"></span>!
</div>
<div>
  Of your <span id="numCalls"></span> calls, <span id="numBackdated"></span> of them are backdated. 
</div>
<div>
  <span id="message"></span>
</div>
`

/**
 * @prop {string} userName
 * @prop {Array<{
 *      createdDateTime: StudioDateTime, 
 *      callDateTime: StudioDateTime
 *      }>} calls 
 * @prop {number} threshold
 */
class TestSimple extends HTMLElement {
    #name = "";
    #calls = [];
    #threshold = 0;

    constructor() {
        super();
        this.attachShadow({ mode: 'open' });
        this.#name = this.userName;
        this.#calls = this.calls;
    }

    connectedCallback() {
        this.shadowRoot.appendChild(template.content.cloneNode(true));
        this.#update();
    }

    set userName(newName) {
        this.#name = newName;
        this.#update();
    }

    set calls(newCalls) {
        this.#calls = newCalls;
        this.#update();
    }

    set threshold(newThreshold) {
        this.#threshold = newThreshold;
        this.#update();
    }

    #update() {
        const numCalls = this.#calls.length;
        const numBackdated = this.#calls.filter(this.#callIsBackdated).length;
        this.#updateSpan("name", this.#name);
        this.#updateSpan("numCalls", numCalls);
        this.#updateSpan("numBackdated", numBackdated);
        if (numBackdated / numCalls < this.#threshold) {
            this.#updateSpan("message", "<div>This is below the threshold. </div><img style='width:200px' src='https://images.pexels.com/photos/45201/kitty-cat-kitten-pet-45201.jpeg' />")
        } else {
            this.#updateSpan("message", "<div>This is above the threshold. </div><img style='width:200px' src='https://images.pexels.com/photos/12428149/pexels-photo-12428149.jpeg?auto=compress&cs=tinysrgb&w=1260&h=750&dpr=1' />")
        }
    }

    #updateSpan(id, value) {
        if (this.shadowRoot.getElementById(id)) {
            this.shadowRoot.getElementById(id).innerHTML = value;
        }
    }

    #callIsBackdated(call) {
        const createdDate = new Date(call.createdDateTime);
        const callDate = new Date(call.callDateTime);
        return [callDate.getFullYear(), callDate.getMonth(), callDate.getDay()] < [createdDate.getFullYear(), createdDate.getMonth(), createdDate.getDay()];
    }
}


customElements.define('test-simple', TestSimple);